<?php
# @Author: Rainer Imb <imb>
# @Date:   2018-05-07 15:01:47
# @Project: consentio mccurdy
# @Last modified by:   consentio
# @Last modified time: 2023-09-13T13:29:10+02:00
# @License: GPL 2.0



/**
 * a libary of functions used for consentio mvc
 * @package consentio mvc
 */

/**
 * @var array building tree
 */
$o_branch = array();
$cur_item = 1;

/**
 * @var array templates cache
 */
$o_templates = array();

/**
 *get parameters in  perl-style
 *@param string name of parameter
 *@return string value of param
 */
function param($name) {
	$param = array_key_exists($name, $_REQUEST) ? $_REQUEST[$name] : "";
	return($param);
	}

/**
 * avoid illegal characters
 * param1 string
 * param2 string type of cleaning: w: words, n: integer, f: float, l: low, e: email, u: url, wr: words restrictive
 * @return string clean value
 */
function clean_var($s, $t) {
	$s = stripcslashes($s);
	$filter = false; $flag = FILTER_FLAG_STRIP_LOW;
	switch ($t) {
		case "n": $filter = FILTER_SANITIZE_NUMBER_INT; $flag = false; break;
		case "f": $filter = FILTER_SANITIZE_NUMBER_FLOAT; $flag = FILTER_FLAG_ALLOW_FRACTION; break;
		case "u": $filter = FILTER_SANITIZE_URL; $flag = false; break;
		case "e": $filter = FILTER_SANITIZE_EMAIL; $flag = false; break;
		case "l": break;
		case "wr": $s = preg_replace("/[^\w\d]/", '', $s); break;
		case "w": $s = preg_replace("/[\/\*\+()\"\'\?]/", '', $s); // HIGH is no good for UTF8
		}
	if ($filter) {$s = filter_var($s, $filter, $flag);}
	else {$s = htmlspecialchars($s);} // PHP 8
	return($s);
	}

/**
 * internationalisation
 * @param string a keyword like "edit"
 * @global the database connection
 * @global the html assoc array
 * @global array of all words build once
 * @return string word in like "Bearbeiten" or "Edit" or "[edit]", if there is no translation
 */
function i18n($tring, $clean = false) {
	global $db, $html, $con, $i18n;
	if(!isset($i18n)) {$i18n = [];}
	$ln = array_key_exists('user', $html) ? $html['user']['language'] : $con['ln'];
	# switch imediately to selected language
	if (param('language')) {$ln = clean_var(param('language'), "w");}
	# load i18n for the first time
	if (!count($i18n)) {
		$groups = "'common'";
		if(array_key_exists('plugin', $html)) {	$groups .= ",'{$html['plugin']}'";}
		$arg = $db->prepare("SELECT `index`, `$ln` FROM `i18n` WHERE `group` IN($groups);");
		$arg->execute();
		while ($data = $arg->fetch(PDO::FETCH_ASSOC)) {
			$i18n[$data['index']] = $data[$ln];
			}
		}
	$w = array_key_exists($tring, $i18n) ? $i18n[$tring] : "[$tring]";
	if ($clean) {$w = ucfirst(preg_replace('/[\[\]]/', '', $w));}
	return $w;
	}

/**
 * reload another language
 * @param string language (de, en, ...)
 */
function switch_i18n($lang) {
	global $db, $html, $i18n;
	$i18n = [];
	$groups = "'common'";
	if(array_key_exists('plugin', $html)) {	$groups .= ",'{$html['plugin']}'";}
	$arg = $db->prepare("SELECT `index`, `$lang` FROM `i18n` WHERE `group` IN($groups);");
	$arg->execute();
	while ($data = $arg->fetch(PDO::FETCH_ASSOC)) {$i18n[$data['index']] = $data[$lang];}
	}

/**
 * add new words of a group
 * @param string name of a group
 * @global array more words
 */
function add_i18n($group) {
	global $db, $html, $con, $i18n;
	$ln = array_key_exists('user', $html) ? $html['user']['language'] : $con['ln'];
	# switch imediately to selected language
	if (param('language')) {$ln = clean_var(param('language'), "w");}
	# load i18n for a new group
	$arg = $db->prepare("SELECT `index`, `$ln` FROM `i18n` WHERE `group` = '$group';");
	$arg->execute();
	while ($data = $arg->fetch(PDO::FETCH_ASSOC)) {$i18n[$data['index']] = $data[$ln];}
	}

/**
 * in seldom cases we need to translate single words
 * in another language as the one who is loaded
 * @param string a keyword like "edit"
 * @global the database connection
 * @return string word "Bearbeiten" if $lang was "de"
 */
function sys_trans($word, $lang) {
	global $db, $con, $i18n;
	if (!isset($i18n)){i18n($word);} // load i18n if it isn't
	if ($lang) {
		if ($lang == $con['ln']) {$word = array_key_exists($word, $i18n) ? $i18n[$word] : "[$word]";}
		elseif (array_key_exists($word."__$lang", $i18n)) {$word = $i18n[$word."__$lang"];}
		else {
			$arg = $db->prepare("SELECT `$lang` FROM `i18n` WHERE `index` = ?;");
			$arg->execute(array($word));
			$d = $arg->fetch(PDO::FETCH_ASSOC);
			$i18n[$word."__$lang"] = $d[$lang] ? $d[$lang] : "[$word]";
			$word = $i18n[$word."__$lang"];
			}
		}
	return $word;
	}

/**
 * format date
 * date
 * lang (en, de)
 * short, medium, long
 */
function i18n_date($date, $lang = false, $type = 'short') {
	global $con;
	if (!$lang) {$lang = $con['ln'];}
	$c = $date;
	if ($type == 'short') {
		switch ($lang) {
			case 'de': $c = date('d.m.Y', strtotime($date)); break;
			default: $c = date('d/m/Y', strtotime($date)); break;
			}
		}
	else { // medium stand alone
		switch ($lang) {
			case 'de':
				$c = date('j', strtotime($date)).'. '.sys_trans(strtolower(date('M', strtotime($date))), $lang).' '.date('Y', strtotime($date));
				break;
			default:
				$c = date('jS', strtotime($date)).' '.sys_trans(strtolower(date('M', strtotime($date))), $lang).' '.date('Y', strtotime($date));
				break;
			}
		}
	if ($type == 'long') {
		switch ($lang) {
			case 'de': $c = sys_trans(strtolower(date('D', strtotime($date))), $lang).", $c"; break;
			default: $c = sys_trans(strtolower(date('D', strtotime($date))), $lang)." $c"; break;
			}
		}
	return $c;
	}

/**
 * multilingual support
 * @return array used languages
 */
function used_langs() {
	global $db, $con;
	$arg = $db->prepare("SELECT * FROM `languages` WHERE `multilingual` = 'on';");
	$arg->execute();
	while ($data = $arg->fetch(PDO::FETCH_ASSOC)) {
		$con['langs'][$data['iso639']] = $data['language'];
		}
	}

/**
 * creates bubbles with help
 * @param string a keyword like "edit"
 * @global the database connection
 * @global array of all bubbles build once
 * @global the html assoc array
 * @return string a html string with javascript
 */
function bubbles($action) {
	global $db, $bubbles, $html;
	$script = "<div class='hidden'><h2>".i18n('help')."</h2>";
	$arg = $db->prepare("select `header`, `body`, `tooltip`, `parent` from `help` where `tooltip` like '$action%' or `tooltip` like 'common%'");
	$arg->execute();
	while ($data = $arg->fetch(PDO::FETCH_ASSOC)) {
		if ($data['header']) {$data['header'] = "<h3>{$data['header']}</h3>";}
		$tooltip_id = preg_replace("/\./", "-", $data['tooltip']);
		$script .= "<div id=\"bubble$tooltip_id\">{$data['header']}{$data['body']}</div>\n";
		$bubbles[$data['tooltip']] = "&nbsp;<a href=\"{$html['myurl']}?_id=help&amp;sid={$html['sid']}&amp;pop=1&amp;parent={$data['parent']}#{$data['tooltip']}\" id='$tooltip_id' data-app='bubble' class='abubble' target='_blank'>".get_icon("info-circle-dblue", "")."</a>";
		$bubbles['_text_'][$data['tooltip']] = $data['body'];
		}
	$script .= "</div>
	<script type='text/javascript'>\$(function(){\$(\"a[data-app='bubble']\").foolTip({'header': '".i18n('help')."'});});</script>";
	return($script);
	}

/**
 * returns a bubble
 * @param string a keyword for help
 * @return string a html string with bubble
 */
function bubble($tooltip) {
	global $bubbles;
	$c = array_key_exists($tooltip, $bubbles) ? $bubbles[$tooltip] : '';
	return $c;
	}

/**
 * returns a help text from bubble
 * @param string a keyword for help
 * @return string a help text
 */
function bubble_text($tooltip) {
	global $bubbles;
	$c = array_key_exists($tooltip, $bubbles) ? $bubbles['_text_'][$tooltip] : '';
	return $c;
	}

/**
 * helping guide
 * @param array the $html
 * @return array modified $html
 */
function help_us($html) {
	global $db, $con;
	$parent = clean_var(param('parent'), 'n');
	$form = [
		"title"			=> i18n('help')." - Consentio {$con['version']}",
		"table"			=> "help",
		"orderby"		=> "sort",
		"html_in_table" => true,
		"myconf"		=> "help",
		"human"			=> "header",
		"html"			=> $html,
		"search"		=> ["fulltext"],
		"filters"		=> ["parent"],
		"table_fields"	=> ["header", "body", "parent", "tooltip"]
		];
	$inputs = [
		"header"		=> ["type" => "text"],
		"body"			=> ["type" => "wysiwyg"],
		"tooltip"		=> ["type" => "text"],
		"pop"			=> ["type" => "free", "value" => param('pop')],
		"parent"		=> [
			"caption"	=> i18n("chapter"),
			"type"		=> "lookup",
			"lval"		=> "graduated",
			"lord"		=> "parentsort",
			"ltable"	=> "help_folder"
			]
		];

	$action = "help";
	if (param('pop')) {$html['gui'] = 4; $action = "help&amp;pop=1";}
	$tree = build_my_tree($parent, $action);
	$obj = new access($form);
	$obj->table($inputs);
	$c = '';
	if ($parent || param('_w')) {
		$table_rows = $obj->table_rows;
		foreach ($table_rows as $data) {
			if ($data['header']) {$c .= "<h3 id='{$data['tooltip']}'>{$data['header']}</h3>";}
			$c .= $data['body'];
			}
		if (param('_w')) {$c = safe_highlight($c, clean_var(param('_w'), 'w'));}
		}
	else {
		$c = bubble_text('common.preface');
		}
	$html['content'] = $obj->searchform.$c;
	$html['formtitle'] = i18n('help')." - Consentio {$con['version']}";
	$html['left'] = parse_in('gui_tree', ['treelist' => $tree]);
	return $html;
	}

function help_tree($open, $action) {
	$tree = build_my_tree($open, $action);
	$tree = parse_in('gui_tree', ['treelist' => $tree]);
	return $tree;
	}

function build_my_tree($open, $action) {
	global $db, $html, $con;
	$tree = "";
	$roots = [
		"parent"	=> 1,
		"pfield"	=> "parent",
		"field"		=> "name",
		"table"		=> "help_folder",
		"sort"		=> "sort",
		"index"		=> 0,
		"open"		=> $open,
		"action"	=> $action
		];
	$tree .= branch($roots);
	return $tree;
	}

/**
 * make a message dialog
 * @param1 string text of message
 * @param1 string icon name as "warning" (png must be in ../icons/48/)
 */
function say_mess($text, $title, $img, $big = false, $after = false) {
	$css = $big ? 'dlg lg' : 'dlg';
	$text_after = $after ? "<div class=\"col w12\">$after</div>" : '';
	$fa = ['warning' => 'exclamation-triangle-gold', 'info' => 'info-circle_blue', 'error' => 'ban-red', 'question' => 'question-circle-blue', 'ok' => 'check-circle-green', 'checked' => 'check-circle-green'];
	$img = array_key_exists($img, $fa) ? $fa[$img] : $img;
	$icon = get_awesome(['icon' => $img, 'alt' => '']);
	if ($title) {$title = "<h2>$title</h2>";}
	$form = <<<TNIRP
	<div class="center">
	<div class="$css clrfx">
	$title
	<div class="dlgtext">
	<div class="col w3 m3 first mess-icon">$icon</div>
	<div class="col w9 m9 last">$text</div>
	$text_after
	</div>
	</div>
	</div>
TNIRP;
	return($form);
	}

/**
 * makes a submenue with tabs and icons
 * @param array consisting a caption, a link, a css class, an icon (in ../icons)
 * @return string <ul> of submenue tabs
 */
function make_submenu($sub) {
	$ul = "\n<ul class='submen clrfx'>";
	foreach ($sub as $a) {
		# dedicated to 16x16px icons
		if ($a[3]){$a[0] = get_icon($a[3], '')."&nbsp;$a[0]";}
		$ul .= "\n\t<li><a href='{$a[1]}' class='{$a[2]}'>{$a[0]}</a></li>";
	}
	$ul .= "\n</ul>";
	return ($ul);
	}

/**
 * sets all to inactive state and one to active
 * @param1 array the submenue array
 * @param2 integer the active tab
 * @return array the modifiedsubmenue array
 */
function toggle_submen($sub, $me) {
	foreach ($sub as $i=>$a) {
		$sub[$i][2] = "inactive";
		}
	$sub[$me][2] = "active";
	return ($sub);
	}

/**
 * test and read data of one register
 * @param array the html assoc array
 * @return array html assoc array with information about plugin and gui
 */
function get_one_register($html) {
	global $db;
	if (!$html['action']) {$html['action'] = "start"; $data['plugin'] = ''; $data['gui'] = ''; $data['tab'] = 0;}
	else {
		$arg = $db->prepare("SELECT `plugin`, `tab`, `gui` FROM `register` WHERE `call` = ?;");
		$arg->execute([$html['action']]);
		$data = $arg->fetch(PDO::FETCH_ASSOC);
		if ($arg->rowCount()) {$html['reg'] = true;}
		else {$data = ['plugin' => '', 'tab' => 0, 'gui' => 1];}
		}
	$html['plugin'] = $data['plugin'] ? $data['plugin'] : $html['action'];
	$html['tab'] = $data['tab'];
	$html['gui'] = $data['gui'] ? $data['gui'] : 1;
	return $html;
	}

function get_board($html, $group) {
	global $db;
	global $html;
	global $user;
	$taint = array();
	$call;
	$tab = array();
	$c = "";
	$q = "SELECT * FROM `register_groups`";
	if ($group != "ALL") {$q .= " WHERE `name` = '$group'";}
	$q .= " ORDER BY `sort`";
	$arg = $db->prepare($q);
	$arg->execute();

	while ($data = $arg->fetch(PDO::FETCH_ASSOC)) {
		$tab[$data['id']] = [
			'name'		=> $data['name'],
			'entries'	=> ''
			];
		$call = $data['id'];
		}

	$q = "SELECT * FROM `register`";
	if ($group == "ALL") {$q .= " WHERE `start` = 'on'";}
	else {$q .= " WHERE `tab` = ? AND `group` = 'on'"; array_push($taint, $call);}
	$q .= " AND `level` <= ? ORDER BY `sort`";
	array_push($taint, $user['level']);
	$arg = $db->prepare($q);
	$arg->execute($taint);

	if ($arg->rowCount()) {
		while ($data = $arg->fetch(PDO::FETCH_ASSOC)) {
			if ($data['call'] != $group) {
				$caption = i18n($data['name'], true);
				$img = $data['icon'] ? $data['icon'] : "default";
				if ($data['image']) {$img = "png-$img";}
				$img = get_big_icon($img, $caption);
				$addon = $data['params'] ? htmlspecialchars($data['params']) : "";
				$tab[$data['tab']]['entries']  .= "<li><a href='{$html['myurl']}?_id={$data['call']}&amp;sid={$html['sid']}$addon'>$img<br>$caption</a></li>";
				}
			}
		}

	foreach ($tab as $k => $v) {
		if ($v['entries']) {
			if ($group == "ALL") {
				$c .= "<li class='t2 mreset menueset clrfx'>\n";
				$bubble = bubble_text('common.'.$v['name']);
				if ($bubble) {$bubble = htmltag('p', '', $bubble);}
				$c .= "<h2>".i18n($v['name'])."</h2>$bubble";
				}
			$c .= "<ul class='menue'>\n{$v['entries']}\n</ul>\n</li>\n";
			}
		}
	if ($group == "ALL") {$c = "<ul class='yo-tiles'>$c</ul>";}
	return $c;
	}

/**
 * builds a tree from table 'register' with a given group
 * @param1 array the html assoc array
 * @param2 string name of the group
 * @param3 string name of the active register
 * @param4 string nested ul as tree-extension
 */
function get_tab_tree($html, $group, $me, $ext) {
	global $db, $html, $user;
	$tree = "";
	$q = "SELECT `register`.* FROM `register`, `register_groups` WHERE `register_groups`.`id` = `register`.`tab` AND `register_groups`.`name` = ?  AND `group` = 'on' AND `level` <= ? ORDER BY `register`.`sort`";
	$arg = $db->prepare($q);
	$arg->execute(array($group, $user['level']));
	if ($arg->rowCount()) {
		while ($data = $arg->fetch(PDO::FETCH_ASSOC)) {
			$state = $me == $data['call'] ? "open" : "closed";
			$sub = $me == $data['call'] ? $ext : "";
			if ($data['call'] == $group) {$state = "parent";}
			$item = [
				"caption"	=> i18n($data['name'], true),
				"link"		=> "{$html['myurl']}?_id={$data['call']}&amp;sid={$html['sid']}",
				"state"		=> $state,
				"sub"		=> $sub
				];
			if (!$data['image']) {$item['reg_icon'] = $data['icon'];}
			$tree .= list_item($item);
			}
		}
	if ($tree) {$tree = parse_in('gui_tree', ['treelist' => $tree]);}
	return $tree;
	}

/**
 * response of ajax requests
 * @param array the html assoc array
 * @return array html assoc array with response
 */
function ajax($html) {
	global $db;
	$html['gui'] = 6;
	$todo = clean_var(param("todo"), "w");
	if ($todo == "autofill" || $todo == "suggest") {
		$limit = 100;
		$html['content'] = "-|0\n";
		$value = param("q");
		$id_field = clean_var(param("id_field"), "w");
		if (!$id_field) {$id_field = "id";}
		$field = clean_var(param("field"), "w");
		$table = clean_var(param("table"), "w");
		$clause = base64_decode(param("clause"));
		if ($clause) {$clause = "AND ".html_entity_decode($clause);}
		$i = 1; $mifis = '';
		while ($i) {
			if (param("mifi$i")) {$mifis .= ', `'.clean_var(param("mifi$i"), "w").'`'; $i++;}
			else {$i = false;}
		}
		if (strpos($field, '|')) {
			$all = explode('|', $field);
			$fields = ' CONCAT(';
			foreach ($all as $v) {$fields .= "`$v`, ' ',";}
			$fields = preg_replace('/,$/', '', $fields);
			$fields .= ") AS `{$all[0]}`";
			$field = $all[0];
			}
		else {$fields = "`$field`";}

		$query = "SELECT DISTINCT $fields, `$id_field` $mifis FROM `$table` WHERE `$field` LIKE ? $clause ORDER BY `$field` LIMIT $limit;";
		if ($todo == "suggest") {$query = "SELECT DISTINCT $fields $mifis FROM `$table` WHERE `$field` like ? $clause ORDER BY `$field` LIMIT $limit;";}
		$value = preg_replace("/\+/", "", $value);
		//error_log("Consentio Notice: ".$query, 0);
		if ($value) {
			$arg = $db->prepare($query);
			$arg->execute(["%$value%"]);
			while ($values = $arg->fetch(PDO::FETCH_NUM)) {
				$output = implode('|', $values);
				$html['content'] .= "$output\n";
				}
			}
		//$html['content'] .= $query;
		}

	# toggle flag of a record
	elseif ($todo == "setAttr") {
		$table = clean_var(param('t'), "w");
		$field = clean_var(param('f'), "w");
		$id = clean_var(param('id'), "n");
		$arg = $db->prepare("SELECT `$field` FROM `$table` WHERE `id` = ?;");
		$arg->execute(array($id));
		$stat = $arg->fetch(PDO::FETCH_ASSOC);
		$stat = $stat[$field] ? "" : "on";
		$arg = $db->prepare("UPDATE `$table` SET `$field` = ? WHERE `id` = ?;");
		$arg->execute(array($stat, $id));
		if ($arg->rowCount() >= 1) {$html['content'] = "OK";}
		else {$html['content'] = i18n('error');}
		}

	// Easy Admin returns info
	elseif ($todo == "eAd") {
		$id = clean_var(param("rid"), "n");
		$ac = clean_var(param("ac"), "w");
		$ti = i18n($ac);
		if ($id) {
			$field = "title"; $table = 'content';
			if ($ac == "announces") {$ti = i18n('teaser'); $field = "title"; $table = "announces";}
			elseif ($ac == "components") {$ti = i18n('components'); $field = "title"; $table = "components";}
			$arg = $db->prepare("SELECT `$field` FROM `$table` WHERE `id` = ?;");
			$arg->execute(array($id));
			if ($id = $arg->fetch(PDO::FETCH_ASSOC)) {$id = "<strong>\"{$id[$field]}\"</strong>";}
			}
		else {$id = "";}
		$html['content'] = "<p>".i18n('edit').":<br />$ti $id</p>";
		}
	elseif ($todo == "smart") {
		$val = clean_var(param("v"), "w");
		$v = explode('|', base64_decode(param('_c')));
		$arg = $db->prepare("UPDATE `{$v[0]}` SET `{$v[1]}` = ? WHERE `id` = ?;");
		$arg->execute(array($val, $v[2]));
		$err = $arg->errorInfo();
		$error = array_key_exists(2, $err) ? $err[2] : '';
		$html['content'] = $error;
		}
	return $html;
	}

/**
 * highlight word in content outside html tags by wrapping a span with class "highlight" arround
 * @param1 string content (HTML)
 * @param2 string whitespace seperated words to be hightlighted
 * @return string HTML with highlighted words
 */
function safe_highlight($content, $words) {
	$arr = preg_split("/\s/", $words, -1, PREG_SPLIT_NO_EMPTY);
	foreach ($arr as $w) {
		if (strlen($w) > 2) {
			$content = preg_replace("/(<[^\<\>]*)($w)([^\<\>]*>)/i", "$1$2~$3", $content);
			$content = preg_replace("/(<[^\<\>]*)($w)([^~][^\<\>]*>)/i", "$1$2~$3", $content);
			$content = preg_replace("/(<[^\<\>]*)($w)([^~][^\<\>]*>)/i", "$1$2~$3", $content);
			$content = preg_replace("/($w)([^~])/i", "<span class='highlight'>$1</span>$2", $content);
			$content = preg_replace("/($w)\~/i", "$1", $content);
			}
		}
	return($content);
	}

function db_image($data) {
	global $db;
	$arg2 = $db->prepare("insert `image_db` (`filename`, `title`, `description`, `parent`, `copyright`, `upload_date`) values (?, ?, ?, ?, ?, ?)");
	$arg2->execute(array($data['filename'], $data['title'], $data['description'], $data['parent'], $data['copyright'], date('Y-m-d H:i:s')));
	}


/**
 * scales an image (JPG, GIF, PNG) in different formats
 * @param string image-file in ../pictures/
 * @global array image dimensions (conf_inc.php)
 */
function scale_image($file) {
	global $image_dim; $ny = 0; $nx = 0;
	$temp = $image_dim;
	unset($temp['cin_ratio']); // ok %P
	$dirs = array_keys($temp);
	$path = $_SERVER['DOCUMENT_ROOT']."/pictures";
	$finfo = finfo_open(FILEINFO_MIME_TYPE);
	$ext = finfo_file($finfo, "$path/$file");
	if ($ext == 'image/svg' || $ext == 'image/svg+xml'){
		foreach ($dirs as $dir) {copy("$path/$file", "$path/$dir/$file");}
		return;
		}
	$img = getimagesize("$path/$file");
	list($x, $y) = $img;
	$image;
	switch ($img['mime']) {
		case "image/jpeg": $image = imagecreatefromjpeg("$path/$file"); break;
		case "image/gif": $image = imagecreatefromgif("$path/$file"); break;
		case "image/png": $image = imagecreatefrompng("$path/$file"); break;
		}
	foreach ($dirs as $dir) {
		if (!file_exists($path.'/'.$dir) && !is_dir($path.'/'.$dir)) {mkdir($path.'/'.$dir, 0777);}
		$ox = 0; $oy = 0; $ex = $x; $ey = $y;
		if ($x > $y) {
			if ($dir == "square") {$ox = intval(($x-$y)/2); $ex = $ey; $nx = $image_dim[$dir]; $ny = $nx;}
			else {
				$nx = $image_dim[$dir]; $ny = intval($nx*($y/$x));
				if ($x < $nx) {$nx = $x; $ny = $y;}
				}
			}
		else {
			if ($dir == "square") {$oy = intval(($y-$x)/2); $ey = $ex; $ny = $image_dim[$dir]; $nx = $ny;}
			else {
				$ny = $image_dim[$dir]; $nx = intval($ny*($x/$y));
				if ($y < $ny) {$nx = $x; $ny = $y;}
				}
			}
		# cinemascope has own rules
		if ($dir == "cinemascope") {
			$ratio = array_key_exists('cin_ratio', $image_dim) ? $image_dim['cin_ratio'] : 0.562; // default 16:9
			$h = intval($x*$ratio);
			if ($y > $h) {
				$oy = intval(($y-$h)/2); $ey = $h;
				$nx = $image_dim[$dir]; $ny = intval($nx*$ratio);
				}
			else {
				$ex = intval($y*(1/$ratio)); $ox = intval(($x-$ex)/2);
				$nx = $image_dim[$dir]; $ny = intval($nx*$ratio);
				}
			}
		$image_new = imagecreatetruecolor($nx, $ny);
		# php 5.5 for better interpolation
		if (function_exists('imagesetinterpolation'))  {imagesetinterpolation($image_new, IMG_NEAREST_NEIGHBOUR);}
		imagealphablending($image_new, false);
		imagesavealpha($image_new, true);
		imagecopyresampled($image_new, $image, 0, 0, $ox, $oy, $nx, $ny, $ex, $ey);
		switch ($img['mime']) {
			case "image/jpeg": imagejpeg($image_new,"$path/$dir/$file"); break;
			case "image/gif": imagegif($image_new,"$path/$dir/$file"); break;
			case "image/png": imagepng($image_new,"$path/$dir/$file"); break;
			}
		}
	}

/**
 * inserts image into db
 * @param string image-file in ../pictures/
 * @param string from plugin
 * @param string title
 * @param string id parent directory
*/
function insert_image_db($file, $from, $title, $p = 1) {
	global $db;
	if ($from != 'images') {
		$desc = i18n($from, true);
		$time = date('Y-m-d H:i:s');
		$arg = $db->prepare("INSERT INTO `image_db`(`filename`, `title`, `description`, `parent`, `upload_date`) VALUES (?,?,?,?,?);");
		$arg->execute(array($file, $title, $desc, $p, $time));
		}
	}

/**
 * upload a file
 * @param string directory
 * @param string filename of a $_FILE
 * @param string u for unique
 */
function upload($dir, $file, $fn) {
	if ($_FILES[$file]['error'] == UPLOAD_ERR_OK) {
		$name = $fn == 'u' ? clean_filename($_FILES[$file]['name'], $dir) : as_filename($_FILES[$file]['name']);
		move_uploaded_file ($_FILES[$file]['tmp_name'], "$dir/$name");
		return $name;
		}
	else {echo $_FILES[$file]['error'];}
	}

/**
 * check user, login and get data
 * @param string the SID
 * @global the database connection
 * @return array the users data
 */
function get_user($sid) {
	global $db;
	if (isset($_SESSION['sid'])) {
		if ($sid != $_SESSION['sid']) {$sid = $_SESSION['sid'];}
		}
	else {$_SESSION['sid'] = $sid;}
	$result = ['feedback' => '', 'level' => 0];
	$query = "SELECT * FROM `user` WHERE `sid` = ?";
	$taint = [$sid];
	if (!$sid) {
		$u = param('user'); $p = param('pwd');
		if ($u) {
			$query = "SELECT * FROM `user` WHERE `name` = ?";
			$taint = array($u);
			}
		else {$query = 0;}
		}
	if ($query) {
		$arg = $db->prepare($query);
		$arg->execute($taint);
		if (!$arg->rowCount()) {
			$result['feedback'] = i18n('false_login');
			error_log("Consentio Notice: login FAILED", 0);
			}
		else {
			$result = $arg->fetch(PDO::FETCH_ASSOC);
			if(!$sid) {
				if (!password_verify($p, $result['pwd'])) {
					$result = ['feedback' => i18n('false_login'), 'level' => 0];
					error_log("Consentio Notice: login FAILED", 0);
					}
				else {
					$result['sid'] = date('YmdHis').sprintf("%04d", $result['id']);
					$arg = $db->prepare("UPDATE `user` SET `sid` = ? WHERE `id` = ?;");
					$arg->execute(array($result['sid'], $result['id']));
					$_SESSION['sid'] = $result['sid'];
					error_log("Consentio Notice: login: {$result['name']}", 0);
					}
				}
			}
		}
	return $result;
	}

/**
 * login form
 * @param array the html assoc array
 * @return array html assoc array with login form
 */
function make_login_form($html) {
	global $con;
	setcookie("coco", '', time()-600, '/');
	session_unset();
	session_destroy();
	$html['sid'] = '';

	$form = [
		"defsize"		=> 20,
		"no_cancel"		=> 1,
		"html"			=> $html,
		"feedback"		=> $html['feedback'],
		"button_text"	=> i18n('login'),
		"button_fa"		=> 'sign-in-alt',
		"remember"		=> false,
		"label_size"	=> 3,
		"submit"		=> 'login' // anything but '_store'
		];
	$inputs = [
		"user"	=> ["type" => "text", "css_class" => 'loose'],
		"pwd"	=> ["type" => "pwd", "css_class" => 'loose'],
		"_id"	=> ["type" => "hidden", "value" => clean_var(param('_id'), 'w')] // remember action
		];
	$obj = new access($form);
	$obj->add_input($inputs);
	$html = $obj->form();
	$html['content'] = say_mess($html['content'], i18n('login'), "key");
	$html['content'] .= htmltag('p', 'class="center small grey"', 'Consentio '.$con['version']);
	$html['gui'] = 5;
	return $html;
	}

/**
 * logout, erases sid
 * @param array the html assoc array
 * @global the database connection
 * @return array html assoc array with logout message
 */
function logout($html) {
	global $db;
	$l = i18n('login');
	$bye = i18n('bye');
	$html['content'] = <<<EOL
	<p class="big center">$bye, <br />{$html['user']['name']}</p>
	<p class='center'><a href="{$html['myurl']}" class="as_button"><i class="fas fa-sign-in-alt fa-lg"></i> $l</a><p>
EOL;

	$arg = $db->prepare("update `user` set `sid` = '' where `id` =?;");
	$arg->execute(array($html['user']['id']));
	setcookie("coco", '', time()-600, '/');
	session_unset();
	session_destroy();
	$html['content'] = say_mess($html['content'], "", "far-handshake");
	$html['sid'] = 0;
	$html['gui'] = 5;
	return $html;
}

/**
 * makes a font awesome icon since 3.101
 * @param string name of fa according to css
 * @param string alternative text
 * @return string i tag for icon or png
 */
function get_icon($icon, $alt, $size = false, $no_title = false) {
	// some translations: former famfamfam icon-> font awesome icon
	$fa = ['all' => 'asterisk-red', 'user-3' => 'user-circle-red', 'user-2' => 'user-circle-orange', 'user-1' =>
	'user-circle-green', 'homepage' => 'home-green', 'languages' => 'commenting-o',
	'world' => 'globe', 'preview' => 'eye', 'add' => 'plus-circle-green',
	'pages' => 'copy', 'world-edit' => 'globe-edit', 'noedit' =>
	'edit-ghost','palette' => 'paint-brush','picture-add' =>
	'image-add','picture-edit' => 'image-edit','picture-empty' =>
	'image-error','picture' => 'image','pictures' => 'images', 'folder_page'
	=> 'folder-go', 'nodelete' => 'times-ghost', 'delete' =>
	'times-red','gallery' => 'file-stack-image', 'exit' => 'sign-out-alt-red', 'info'
	=> 'info-circle-dblue', 'vcard_add' => 'address-card-add', 'vcard' =>
	'address-card', 'serial_page' => 'file-excel-orange', 'page' => 'file-alt',
	'sortnew' => 'sync-dblue', 'articles' => 'newspaper', 'package_go' => 'upload-green', 'parent' =>
	'level-up-alt-green', 'properties' => 'far-hand-point-up', 'plugin' => 'puzzle-piece', 'package-refresh' => 'sync',
	'update' => 'sync', 'mirror' => 'arrows-alt-h', 'meta' => 'code', 'warning' => 'exclamation-triangle-gold', 'edit' => 'edit-gold'];
	$aicon = array_key_exists($icon, $fa) ? $fa[$icon] : $icon;
	$fas = ['icon' => $aicon, 'alt' => $alt];
	if ($size) {$fas['size'] = $size;}
	if ($no_title) {$fas['no_title'] = true;}
	// png icon as before 3.101
	if (preg_match("/^png-/", $icon)) {
		$icon = preg_replace("/^png-/", '', $icon);
		if (!file_exists("../co_icons/$icon.png")) {$icon = "todo";}
		$c = "<img src='/co_icons/$icon.png' alt='$alt' title='$alt' width='16' height='16' class='icon' />";
		}
	else {$c = get_awesome($fas);}
	return $c;
	}

/**
 * makes a flag
 * @param string png-file of ../co_icons/flag
 * @param string alternative text
 * @return string image tag for icon
 */
function get_flag_by_country($icon, $alt) {
	if (!file_exists("../co_icons/flags/$icon.png")) {$icon = "unknown";}
	return "<img src='/co_icons/flags/$icon.png' alt='$alt' title='$alt' width='16' height='11' class='icon flag' />";
	}

/**
 * makes a big icon
 * @param string png-file of ../icons/48 or fa-icon
 * @param string alternative text
 * @return string image tag for icon or fa-icon
 */
function get_big_icon($icon, $alt) {
	$fas = ['icon' => $icon, 'size' => '4x', 'alt' => $alt];
	// png icon as before 3.101
	if (preg_match("/^png-/", $icon)) {
		$icon = preg_replace("/^png-/", '', $icon);
		if (!file_exists("../co_icons/48/$icon.png")) {$icon = "unknown";}
		$c = "<img src='/co_icons/48/$icon.png' alt='$alt' title='$alt' width='48' height='48' class='icon' />";
		}
	else {$c = get_awesome($fas);}
	return $c;
	}

/**
 * makes a font awesome tag
 * @param array definitions 'icon' and 'alt' mandatory, optional: set, size, pull
 * @return string image tag for icon or fa-icon
 */
function get_awesome($fas) {
	if (!array_key_exists('set', $fas)) {$fas['set'] = 'fas';}
	if (!array_key_exists('size', $fas)) {$fas['size'] = 'lg';}
	$pull = '';
	if (array_key_exists('pull', $fas)) {$pull = 'fa-pull-'.$fas['pull'];}
	if (preg_match("/-stack-/", $fas['icon'])) {
		list ($ic, $st) = explode('-stack-', $fas['icon']);
		$fas['icon'] = $ic; $fas['stack'] = $st;
		}
	if (preg_match("/^fab-/", $fas['icon'])) {$fas['set'] = 'fab'; $fas['icon'] = preg_replace("/^fab-/", '', $fas['icon']);}
	if (preg_match("/^far-/", $fas['icon'])) {$fas['set'] = 'far'; $fas['icon'] = preg_replace("/^far-/", '', $fas['icon']);}
	if (preg_match("/^flip-/", $fas['icon'])) {$pull .= ' fa-flip-horizontal'; $fas['icon'] = preg_replace("/^flip-/", '', $fas['icon']);}
	if (preg_match("/^rotate-/", $fas['icon'])) {$pull .= ' fa-rotate-90'; $fas['icon'] = preg_replace("/^rotate-/", '', $fas['icon']);}
	if (preg_match("/[_-]add$/", $fas['icon'])) {$fas['sub']['icon'] = 'plus-circle'; $fas['sub']['color'] = 'green'; $fas['icon'] = preg_replace("/[_-]add$/", '', $fas['icon']);}
	if (preg_match("/[_-]edit$/", $fas['icon'])) {$fas['sub']['icon'] = 'pen-square'; $fas['sub']['color'] = 'green'; $fas['icon'] = preg_replace("/[_-]edit$/", '', $fas['icon']);}
	if (preg_match("/[_-]delete$/", $fas['icon'])) {$fas['sub']['icon'] = 'minus-circle'; $fas['sub']['color'] = 'red'; $fas['icon'] = preg_replace("/[_-]delete$/", '', $fas['icon']);}
	if (preg_match("/[_-]img$/", $fas['icon'])) {$fas['sub']['icon'] = 'image'; $fas['sub']['color'] = 'green'; $fas['icon'] = preg_replace("/[_-]img$/", '', $fas['icon']);}
	if (preg_match("/[_-]go$/", $fas['icon'])) {$fas['sub']['icon'] = 'arrow-alt-circle-right'; $fas['sub']['color'] = 'green'; $fas['icon'] = preg_replace("/[_-]go$/", '', $fas['icon']);}
	if (preg_match("/[_-]error$/", $fas['icon'])) {$fas['sub']['icon'] = 'exclamation-circle'; $fas['sub']['color'] = 'gold'; $fas['icon'] = preg_replace("/[_-]error$/", '', $fas['icon']);}

	$alt_type = array_key_exists('no_title', $fas) ? 'alt' : 'title';
	$color = '';
	foreach (['green', 'blue', 'dblue', 'red', 'gold', 'orange', 'purple', 'brown', 'ghost', 'white'] as $c) {
		if (preg_match("/[_-]$c$/", $fas['icon'])) {$color = $c; $fas['icon'] = preg_replace("/[_-]$c$/", '', $fas['icon']);}
		}

	if ($fas['icon'] == "default") {$fas['icon'] = 'puzzle-piece';}
	$fa =  "<i class=\"{$fas['set']} fa-{$fas['icon']} fa-{$fas['size']} $pull $color\" $alt_type=\"{$fas['alt']}\"></i>";
	if (array_key_exists('stack', $fas)) {
		$size = ' fa-lg';
		if ($fas['size'] == 'lg' || $fas['size'] == 's') {$size = ' fa-xs';}
		$fa = "<span class=\"fa-stack fa-{$size}\" $alt_type=\"{$fas['alt']}\"><i class=\"{$fas['set']} fa-{$fas['icon']} fa-stack-2x $color\"></i> <i class=\"fas fa-{$fas['stack']} fa-stack-1x ghost\"></i></span>";
		}
	if (array_key_exists('sub', $fas)) {
		$size = ' fa-lg';
		if ($fas['size'] == 'lg' || $fas['size'] == 's') {$size = ' fa-xs';}
		$cclass = array_key_exists('color', $fas['sub']) ? " {$fas['sub']['color']}" : '';
		if (!array_key_exists('set', $fas['sub'])) {$fas['sub']['set'] = 'fas';}
		$fa .= "<i class=\"{$fas['sub']['set']} fa-{$fas['sub']['icon']}$size imo$cclass\"></i>";
		}
	if (array_key_exists('subtext', $fas)) {
		$fa .= "<span class=\"count\">{$fas['subtext']}</span>";
		}
	return $fa;
	}

/**
 * create a thumbnail
 * @param string image-file of ../pictures
 * @param string id-attribute
 * @param string subdir of ../picture (thumbs or square makes sense)
 * @return string image tag for thumbnail
 */
function get_thumb($file, $id, $dir, $data = '', $alt = '', $stamp = false) {
	global $image_dim;
	$image = "<img id='$id' src='/co_icons/48/picture-empty.png' alt=' '>";
	if ($file && file_exists("../pictures/$dir/$file")) {
		list($x, $y) = getimagesize("../pictures/$dir/$file");
		if (!$x) {$x = $image_dim[$dir];} // svg
		if (!$alt) {$alt = $file;}
		if ($stamp) {$file .= '?'.time();}
		$image = "<img id='$id' src='/pictures/$dir/$file' width='$x' height='$y' alt='$alt' title='$alt' $data />";
		}
	return $image;
	}

/**
 * test if an image is landscape or portrait an has a minimum width or height
 * @param string image-file of ../pictures
 * @param string subdir of ../picture
 * @param string L for landscape P for portrait
 * @param number min-width
 * @param number min-height
 * @return boolean true or false
 */
function test_img($file, $dir, $rel, $w, $h) {
	$check = true;
	if ($file && file_exists("../pictures/$dir/$file")) {
		list($x, $y) = getimagesize("../pictures/$dir/$file");
		if ($rel == 'P' && $x > $y) {$check = false;}
		if ($rel == 'L' && $x < $y) {$check = false;}
		if ($w && $x < $w) {$check = false;}
		if ($h && $y < $h) {$check = false;}
		}
	else {$check = false;}
	return $check;
	}

function get_country_name($val) {
	global $con, $db, $html;
	if (!$val) {return '';}
	$lang = $html['user']['language'] ? $html['user']['language'] : $con['ln'];
	$query = "SELECT `$lang` FROM `countries` WHERE `iso3166` = ? LIMIT 1";
	$arg = $db->prepare($query);
	$arg->execute([$val]);
	//error_log("Consentio Notice: $lang -$val-", 0);
	$data = $arg->fetch(PDO::FETCH_ASSOC);
	return $data[$lang];
	}

function parse_in($key, $data) {
	global $db;
	global $o_templates;
	$out = "";
	if (!array_key_exists($key, $o_templates)) {
		$arg = $db->prepare("SELECT `code` FROM `html_snippets` where `key` = ? limit 1;");
		$arg->execute(array($key));
		$template = $arg->fetch(PDO::FETCH_ASSOC);
		$out = $template['code'];
		$o_templates[$key] = $out;
		}
	else {$out = $o_templates[$key];}
	if ($out) {
		$out = preg_replace_callback("/&&&&i18n_(\w+)&&&&/", function($m) {return i18n($m[1]);}, $out);
		$out = preg_replace_callback("/&&&&(\w+)&&&&/", function($m) use ($data){
			return array_key_exists($m[1], $data) ? $data[$m[1]] : '';
			}, $out);
		$out = "\n<!-- snippet: $key -->\n $out\n<!-- end snippet $key -->\n";
		}
	else {$out = "Template $key has not been found!";}
	//echo "-$key-"; // debug
	return $out;
	}

/**
 * writes settings
 * @param array $data the input
 */
function write_setting($data)	{
	global $db, $con;
	$query = "UPDATE `{$data['table']}` SET `value` = ? WHERE `key` = ? AND `group` = ? AND `lang` = ? AND `rule` = ?;";
	$arg = $db->prepare("SELECT `id` FROM `{$data['table']}` WHERE `key` = ? AND `group` = ? AND `lang` = ? AND `rule` = ?;");
	$arg->execute(array($data['key'], $data['group'], $data['lang'], $con['rule']));
	if (!$arg->rowCount()) {
		$query = "INSERT INTO `{$data['table']}` SET `value` = ?, `key` = ?, `group` = ?, `lang` = ?, `rule` = ?;";
		}
	$arg = $db->prepare($query);
	$arg->execute(array($data['value'], $data['key'], $data['group'], $data['lang'], $con['rule']));
	}

function read_sql($sql) {
	global $db;
	$allow = ['INSERT', 'UPDATE'];
	$lines = explode("\n", $sql);
    $query = "";
    $err = "";
    foreach($lines as $line){
		$line = trim($line);
		if($line == "" OR preg_match('/^\s*(#|--\s)/sU', $line)){}
		else {
            $query .= $line;
            if (substr(rtrim($query), -1) == ';'){
				preg_match('/^\s*(\w*)\s.*/', $query, $m);
				if (in_array(strtoupper($m[1]), $allow)) {
					$arg = $db->prepare($query);
					$check = $arg->execute();
					if ($check === false) {
						$err = $arg->errorInfo();
						$error = array_key_exists(2, $err) ? $err[2] : "";
						$err .= "$error ";
						}
					}
				else {$err .= "$error ";}
				$query = "";
				}
			}
		}
	return $err;
	}

/**
 * replaces common umlauts and accents like Ö to Oe, transliterates cyrillic letters
 * @param string text like "Größe"
 * @return string like Groesse
 */
function ansitrans($string) {
	// it is a latin 1 issue so we decode
	$newstring = utf8_decode($string);
	if (preg_match('/^\?/',$newstring)){
		// it must be something different than latin1
		// try cyrillic
		$cyr  = array('а','б','в','г','д','е','ё','ж','з','и','й','к','л','м','н','о',
		'п','р','с','т','у','ф','х','ц','ч','ш','щ','ъ', 'ы','ь', 'э', 'ю','я',
		'А','Б','В','Г','Д','Е','Ё','Ж','З','И','Й','К','Л','М','Н','О',
		'П','Р','С','Т','У','Ф','Х','Ц','Ч','Ш','Щ','Ъ', 'Ы','Ь', 'Э', 'Ю','Я' );
		$trans = array( 'a','b','v','g','d','e','yo','zh','z','i','y','k','l','m','n','o',
		'p','r','s','t','u','f' ,'h' ,'ts' ,'ch','sh' ,'shch' ,'ie', 'y', '-', 'e' ,'iu' ,'ia',
		'A','B','V','G','D','E','Yo','Zh','Z','I','Y','K','L','M','N','O',
		'P','R','S','T','U','F' ,'H' ,'Ts' ,'Ch','Sh' ,'Shch' ,'Ie' ,'Y' ,'-', 'E', 'Iu' ,'Ia' );
		$string = str_replace($cyr, $trans, $string);
		}
	else {
		$lat  = array('à', 'á', 'â', 'ã', 'ä', 'å', 'æ', 'ç', 'è', 'é', 'ê', 'ë',
		'ì', 'í', 'î', 'ï', 'ð', 'ñ', 'ò', 'ó', 'ô', 'õ', 'ö', 'ø', 'ù', 'ú',
		'û', 'ü', 'ý', 'þ', 'ÿ',
		'À', 'Á', 'Â', 'Ã', 'Ä', 'Å', 'Æ', 'Ç', 'È', 'É', 'Ê', 'Ë', 'Ì', 'Í', 'Î', 'Ï',
		'Ð', 'Ñ', 'Ò', 'Ó', 'Ô', 'Õ', 'Ö', 'Ø', 'Ù', 'Ú', 'Û', 'Ü', 'Ý', 'Þ', 'ß',
		'£', '¥', '€', 'Š', 'š', 'Ž', 'ž', 'Œ', 'œ', 'Ÿ');
		$trans = array('a', 'a', 'a', 'a', 'ae', 'oa', 'ae', 'c', 'e', 'e', 'e', 'e',
		'i', 'i', 'i', 'i', 'th', 'n', 'o', 'o', 'o', 'o', 'oe', 'o', 'u', 'u',
		'u', 'ue', 'y', 'th', 'y',
		'A', 'A', 'A', 'A', 'Ae', 'Oa', 'Ae', 'C', 'E', 'E', 'E', 'E', 'I', 'I', 'I', 'I',
		'Th', 'N', 'O', 'O', 'O', 'O', 'Oe', 'O', 'U', 'U', 'U', 'Ue', 'Y', 'Th', 'ss',
		'_Pound', '_Yen', '_Euro', 'Sch', 'sch', 'Zsch', 'zsch', 'Oe', 'oe', 'Y');
		$string = str_replace($lat, $trans, $string);
		}
	# remove the rest
	$string = preg_replace('/[^\w\d\._-]/', '', $string);
	return ($string);
	}

/** don't put a folder in his own folder or a subfolder of the folder
 * @param1 string name of table
 * @param2 string id of record (e.g. a folder)
 * @param3 string name of parent
 * @return string a where clause
 */
function check_dependencies($table, $id, $parent) {
	$notin = $id;
	list($notin, $table, $id, $parent) = _next_child($notin, $table, $id, $parent);
	return ("`id` not in ($notin)");
	}

/**
 * related to check_dependencies
 * @global database connection
 */
function _next_child($notin, $table, $id, $parent) {
	global $db;
	$arg = $db->prepare("select `id` from `$table` where `$parent` = '$id'");
	$arg->execute();
	while ($nid = $arg->fetch(PDO::FETCH_NUM)) {
		$notin .= ", $nid[0]";
		list($notin, $table, $id, $parent) = _next_child($notin, $table, $nid[0], $parent);
		}
	return array($notin, $table, $id, $parent);
	}

/*
 * let it count 10, 20, 30,... for better resorting
 * @param1 string table
 * @param2 string name of datafield group
 * @param3 string value of group
 */
function reorder($table, $group, $value, $order = 'sort') {
	global $db;
	$q = "SELECT `id` FROM `$table`";
	if ($group && $value) {$q .= " WHERE `$group` = '$value'";}
	$q .= " ORDER BY `$order`;";
	$arg = $db->prepare($q);
	$arg->execute();
	$i = 10;
	while ($id = $arg->fetch(PDO::FETCH_ASSOC)) {
		$arg2 = $db->prepare("UPDATE `$table` SET `sort` = '$i' WHERE `id` = ?;");
		$arg2->execute(array($id['id']));
		$i += 10;
		}
	}

/**
 * creates a hierarchically structured order for cascaded groups
 * @param string table (must contain fields: id, sort, graduated, parentsort)
 * @param string name of datafield which contains the title, the name
 * @param string name of datafield which contains the group
 */
function reconf_parent($table, $name, $group) {
	_next_parent(1, 0, "", $table, $name, $group);
	}

/**
 * depends on reconf_parent
 */
function _next_parent ($count, $parent, $pre, $table, $name, $group) {
	global $db;
	$query = "select `id`, `$name` from `$table` where `$group` = '$parent' and `$name` <> '' order by `sort`";
	$arg = $db->prepare($query);
	$arg->execute();
	while ($data = $arg->fetch(PDO::FETCH_ASSOC)) {
		$arg2 = $db->prepare("update `$table` set `graduated` = '$pre {$data[$name]}', `parentsort` = '$count' where `id` = ?");
		$arg2->execute(array($data['id']));
		$count++;
		$count = _next_parent($count, $data['id'], $pre."—", $table, $name, $group);
		}
	return($count);
	}

/**
 * makes a list item for the ul tree
 * @param array the items data
 * @return string html list item
 */
function list_item($item) {
	global $cur_item;
	$bubble = array_key_exists('bubble', $item) ? $item['bubble'] : " ";
	$state = array_key_exists('state', $item) ? $item['state'] : "open";
	$sub = array_key_exists('sub', $item) ? $item['sub'] : "";
	$color = array_key_exists('color', $item) ? $item['color'] : false;
	$emph = array_key_exists('emph', $item) ? $item['emph'] : false;
	$collapse = 'checked="checked" ';
	switch ($state) {
		case 'closed': $collapse = ""; $state = 'folder'; break;
		case 'open': $state = 'folder-open'; $emph = true; break;
		case 'config': $state = 'wrench'; break;
		case 'langs': $state = 'comment'; break;
		case 'css': $state = 'css3'; break;
		case 'caution': $state = 'warning'; break;
		case 'all': $state = 'all'; break;
		case 'parent': $state = 'parent'; break;
		default: $state = $state; break;
		}
	if (array_key_exists('reg_icon', $item)) {$state = $item['reg_icon'];}
	if ($color) {$state .= '-'.$color;}
	$icon = get_icon($state, $item['caption'], 's', true);
	if ($emph) {$item['caption'] = htmltag('strong', '', $item['caption']);}
	if ($item['link']) {$item['caption'] = "<a href='{$item['link']}'>{$item['caption']}</a>";}
	$li = "\t<li>";
	if ($sub) {$li .= "<input type='checkbox' id='it$cur_item' $collapse /><label class='tree_label' for='it$cur_item'>$icon</label>";}
	else {$li .= "<span class='tree_label' for='it$cur_item'>$icon</span>";}
	$li .= "\t&nbsp;{$item['caption']}$bubble$sub</li>\n";
	$cur_item++;
	return ($li);
	}

/**
 * a branch of a tree
 *
 */
function branch($roots) {
	global $db,	$html, $o_branch;
	$free = array_key_exists('free', $roots) ? $roots['free'] : '';
	$qtype = $roots['table'] == 'folder_view' ? ', `type`' : '';
	$exclusive = array_key_exists('exclusive', $roots) ? " AND `type` = '{$roots['exclusive']}'" : '';
	$arg = $db->prepare("SELECT `id`, `{$roots['field']}`$qtype FROM `{$roots['table']}` WHERE `{$roots['pfield']}` = '{$roots['parent']}'$exclusive ORDER BY `{$roots['sort']}`;");
	$arg->execute();
	$err = $arg->errorInfo();
	$error = array_key_exists(2, $err) ? $err[2] : "";
	$content = $error ? $error : "";
	$color = false;
	$i = $roots['index'];
	$roots['index']++;
	while ($data = $arg->fetch(PDO::FETCH_ASSOC)) {
		$roots['parent'] = $data['id'];
		$state = "closed";
		if (array_key_exists('type', $data)) {
			switch ($data['type']) {
				case "rubric": $color = 'blue'; break;
				case "webshop":
				case "goodsgroup": $color = 'red'; break;
				default: $color = false;
				}
			}
		if ($data['id'] == $roots['open']) {
			$o_branch[$i] = 1; # mark open folder
			$state = "open";
			}
		else {if (!array_key_exists($i, $o_branch)) {$o_branch[$i] = 0;}}
		$sub = branch($roots);
		if (array_key_exists($i+1, $o_branch)){if ($o_branch[$i+1]) {$state = "open"; $o_branch[$i+1] = 0; $o_branch[$i] = 1;}}
		if ($sub){$sub = "<ul>$sub</ul>\n";}
		$item = [
			"caption"	=> $data[$roots['field']],
			"link"		=> "{$html['myurl']}?_id={$roots['action']}&amp;sid={$html['sid']}&amp;{$roots['pfield']}={$data['id']}$free",
			"state"		=> $state,
			"sub"		=> $sub,
			"color"		=> $color
			];
		$content .= list_item($item);
		}
	return $content;
	}

/**
 * like branch without branches
 *
 */
function solo_tree($roots) {
	global $db;
	global $html;
	$arg = $db->prepare("SELECT `id`, `{$roots['field']}` FROM `{$roots['table']}` ORDER BY `{$roots['sort']}`;");
	$arg->execute();
	$err = $arg->errorInfo();
	$error = array_key_exists(2, $err) ? $err[2] : "";
	$content = $error ? $error : "";
	while ($data = $arg->fetch(PDO::FETCH_ASSOC)) {
		$state = "closed";
		if ($data['id'] == $roots['open']) {$state = "open";}
		$item = [
			"caption"	=> $data[$roots['field']],
			"link"		=> "{$html['myurl']}?_id={$roots['action']}&amp;sid={$html['sid']}&amp;{$roots['pfield']}={$data['id']}",
			"state"		=> $state
			];
		$content .= list_item($item);
		}
	return $content;
	}

/**
 * tree of directories
 * @param1 string called action
 * @param2 string base directory
 * @param3 actual path
 * @return string the list for the directory tree *
 */
function next_dir($me, $base, $path) {
	global $html;
	$tree = "";
	if ($path) {$path .= "/";}
	$files = scandir("$base/$path");
	$open = preg_replace("/^\//", '', param('_d'));
	foreach ($files as $file) {
		$dfile = "$base/$path$file";
		if (is_dir($dfile)) {
			if (!in_array($file, ['.', '..'])) {
				$subtree = next_dir($me, $base, "$path$file");
				if ($subtree) {$subtree = "<ul>$subtree</ul>";}
				$state = "closed";
				$clean = preg_replace("/\//", '\/', $path.$file);
				if (preg_match("/^$clean/", $open)) {$state = "open";}
				$item = [
					"caption"	=> $file,
					"link"		=> "{$html['myurl']}?_id=$me&amp;sid={$html['sid']}&amp;_d=/$path$file&amp;_docs=1",
					"state"		=> $state,
					"color"		=> 'green',
					"sub"		=> $subtree
					];
				$tree .= list_item($item);
				}
			}
		}
	return ($tree);
	}

function dir_select($base) {
	global $html;
	$dirs = [$base];
	$dirs = next_dir_list($base, $dirs);
	foreach ($dirs as $k => $v) {
		$dirs[$k] = str_replace($html['dir'], '', $v);
		}
	return $dirs;
	}

function next_dir_list($dir, $dirs) {
	$files = scandir($dir);
	foreach ($files as $file) {
		$dfile = "$dir/$file";
		if (is_dir($dfile)) {
			if (!preg_match('/^\./', $file)) {
				array_push($dirs, $dfile);
				$dirs = next_dir_list($dfile, $dirs);
				}
			}
		}
	return $dirs;
	}

/*
 * converts a html formatted text
 * to plain text
 * @param string html
 * @return string text
 */
function html2plain ($h, $wrap = 70) {
	$h = strip_tags($h, '<br><br/><h1><h2><h3><h4><h5><h6><p><ul><ol><li><a><img><img/><span>');
	$h = preg_replace('/[\n\r\t]/', '', $h); // no newlines, no tabs
	$h = preg_replace('/<h\d[^>]*>(.*?)<\/h\d>/i', "\n# ".'${1}'."\n", $h); // headers
	$h = preg_replace('/<[o|u]l[^>]*>(.*?)<\/[o|u]l>/i', "\n".'${1}'."\n", $h); // ol ul
	$h = preg_replace('/<li[^>]*>(.*?)<\/li>/i', "+ ".'${1}'."\n", $h); // li
	$h = preg_replace('/<p[^>]*?>(.*?(?=<\/p>))<\/p>/i', "\n".'${1}'."\n", $h); // paragraph
	$h = preg_replace('/<span[^>]*?>(.*?(?=<\/span>))<\/span>/i', '${1}'." ", $h); // span
	$h = preg_replace('/<br[^>]*>/i', "\n", $h); // break
	$h = preg_replace('/[\n]{3,}/', "\n\n", $h); // 2 newline only
	$h = preg_replace('/<a [^>]*href="([^"]+)"[^>]*>(.*?)<\/a>/i', '${2} (${1})' , $h); //  save href
	$h = preg_replace('/<img [^>]*alt="([^"]+)"[^>]*\/?>/i', ' ${1} ' , $h); // img save alt
	$h = strip_tags($h); // something left? could be...
	$h = preg_replace('/ {2,}/', ' ', $h); // only one space
	$h = html_entity_decode($h);
	if ($wrap) {$h = wordwrap($h, $wrap, "\n");}
	return $h;
	}

/*
 * easy formatting for bigtext
 * @param string without format but wiki-like controls
 * @return string formattes
 */
function wiki_text ($t) {
	// abbreviation attribute
	// {ab:et cetera{etc.}} gets <abbr title="et cetera">etc.</abbr>
	$t = preg_replace('/{ab:([^{]+){(([^}]|\}[^}])+)}}/', "<abbr title=\"$1\">$2</abbr>", $t);
	// {ac:United Nations Organisation{UNO}} gets <acronym title="United Nations Organisation">UNO</acronym>
	$t = preg_replace('/{ac:([^{]+){(([^}]|\}[^}])+)}}/', "<acronym title=\"$1\">$2</acronym>", $t);
	// language attribute
	// {en{English}} gets <span lang="en">English</span>
	$t = preg_replace('/{(\w{2}){(([^}]|}[^}])+)}}/', "<span lang=\"$1\">$2</span>", $t);
	// formats
	$t = preg_replace('/\*\*(([^*]|\*[^*])+)\*\*/', "<strong>$1</strong>", $t); // **foo** gets strong
	$t = preg_replace('/\+\+(([^+]|\+[^+])+)\+\+/', "<em>$1</em>", $t); // ++foo++ gets emphasized
	// some usefull substitutions
	$t = preg_replace('/---/', "&mdash;", $t);
	$t = preg_replace('/--/', "&ndash;", $t);
	$t = preg_replace('/\(c\)/', "&copy;", $t);
	$t = preg_replace('/\(r\)/', "&reg;", $t);
	$t = preg_replace('/->/', "&rarr;", $t);
	$t = preg_replace('/<-/', "&larr;", $t);
	// newline to break
	$t = nl2br($t);
	return $t;
	}

/**
 * extracts plain text from a body content
 * ignores headers and divs
 * @param string html
 * @return string plain text
 */
function co_strip_tags($tring) {
	if ($tring) {
		// try to kill divs
		$val = preg_replace('/<div[^>]*>.*<\/div>/isU', ' ', $tring);
		//echo strlen($tring)-strlen($val);
		if (strlen($val) < 20) {$val = $tring;} // reset if content is in div
		// kill header (tag and content)
		$val = preg_replace('/<h[^>]*>.*<\/h\d>/', ' ', $val);
		// kill script
		$val = preg_replace('/<script[^>]*>.*<\/script>/is', ' ', $val);
		// remove tags
		$val = preg_replace('/<\/?[^>]*>/', ' ', $val);
		$val = preg_replace('/\s\s+/', ' ', $val);
		}
	else {$val = '';}
	return $val;
	}

/**
 * cuts a string $len + next whitespace...
 * @param1 string some long text
 * @param2 string a number
 * @return string a short text
 * @return string the long text as title
 */
function str_eclipse($tring, $len) {
	$val = ''; $title = '';
	if ($tring) {
		$val = cut_text($tring, $len);
		$title = strlen($tring) > strlen($val) ? " title='$tring'" : "";
		}
	return array($val, $title);
	}

/**
 * finds a passage in a string
 * @param1 string some long text
 * @param3 string words that should be found
 * @param2 string a number
 * @return string a short text
 */
function str_eclipse_with($tring, $words, $len) {
	$str = explode(" ", $words);
	$val = $tring;
	foreach ($str as $w) {
		$pos = stripos($tring, $w);
		if ($pos !== false) {
			$l = $pos-(intval($len/2)+5);
			$l = $l < 0 ? 0 : $l;
			$val = substr($tring, $l, $len+20);
			if ($l) {$val = preg_replace('/^\S+/', '...', $val);}
			}
		}
	// nothing found?
	if (strlen($val) > $len) {$val = cut_text($val, $len);}
	return $val;
	}

/**
 * cuts a string $len + next whitespace...
 * @param1 string some long text
 * @param2 string a number
 * @return string a short text
 */
function cut_text($tring, $len, $sign = '...') {
	$val = '';
	if ($tring) {
		$val = substr($tring, 0, ($len*2));
		$val = preg_replace("/[\n\r]/", " ", $val);
		$val = preg_replace('/\s\s+/', ' ', $val);
		$p = "/([\S\W]{".$len."})(\S+)[\W\S]*/";
		$val = preg_replace($p, "$1$2", $val);
		if (strlen($tring) > strlen($val)) {$val .= $sign;}
		}
	return $val;
	}

/**
 * detects if a text has code or not
 * @param1 string some long text
 * @return boolean true or false
 */
function is_html($tring) {
	$a = strlen($tring);
	$b = strlen(strip_tags($tring));
	$c = $a == $b ? false : true;
	return $c;
	}

/**
 * the HTML5 upload script
 * @param string file-input to extend
 * @return string the script
 */
function uploadscript($field, $accept = '') {
	$max1 = size_bytes(ini_get('upload_max_filesize'));
	$max2 = ini_get('max_file_uploads');
	$max3 = ini_get('post_max_size');
	$maxinf1 = "max. ".sizeme($max1);
	$maxinf2 = i18n('reduce_img').'<br />'.preg_replace("/_-''-_/", $max2, i18n('too_much'));
	$uploadscript = <<<TNIRP
<script src="/js/jQup5.min.js" type="text/javascript"></script>
<script type="text/javascript">\$(document).ready(function () {\$('#$field').upload5({
	'max': '$max1',
	'max_files': '$max2',
	'max_post': '$max3',
	'instruct': '<i class="fas fa-cloud-upload-alt fa-5x ghost"></i><br />&&&&dragndrop&&&&',
	'submit': '_store',
	'destroy': true,
	'toobig': '<span class="red">&&&&skipped&&&& ($maxinf1)</span>',
	'toomuch': '<span class="red">$maxinf2</span>',
	'skipped': '<span class="red">&&&&skipped&&&& $accept</span>',
	'fallback': '<p class="red">&&&&no_html5&&&& &&&&no_multiple&&&&</p>',
	'queue': '&&&&queue&&&&',
	'files': '&&&&files&&&&',
	'ready': '&&&&ready&&&&'
});});</script>
TNIRP;
	return preg_replace_callback("/&&&&(\w+)&&&&/", function($m){return i18n($m[1]);}, $uploadscript);
	}

function as_filename($name) {
	$name = preg_replace("/^\s/", "", $name);
	$name = preg_replace("/\s/", "-", $name);
	$name = ansitrans($name);
	$name = substr($name, 0, 120); // thank you mister hein!
	return $name;
	}

function clean_filename($filename, $dir) {
	# make sure we have an unique site name
	# no use of basename or pathinfo because charset and locale quirks
	# (too much for this usage)
	$fp = preg_split('/\./', $filename);
	$ext = array_pop($fp);
	$name = join('.', $fp);
	$name = as_filename($name);
	$i = 1;
	while (file_exists("$dir/$name.$ext")) {
		$name = preg_replace("/_\d+$/", "", $name);
		$name .= "_$i";
		$i++;
		}
	return("$name.$ext");
	}

/**
 * just open a file and return content as string
 * @param1 string path after docroot and filename
 * @return string filecontent
 */
function open_txt($file) {
	$txt = file_get_contents($_SERVER['DOCUMENT_ROOT']."/$file");
	return $txt;
	}

/**
 * very simple function to create a html tag
 * @param1 string the html-tag like "p", "ul" etc.
 * @param2 mixed string for attributes
 * 		like 'id="hurz"'
 * 		or array like ['id' => 'hurz', 'class' => 'habicht']
 * 		or json like {"id": "hurz", "class": "habicht"}
 * @param3 string the string that should be surrounded with a tag
 * @return string the tag
 */
function htmltag($tag, $attr, $string) {
	$self_closing = ['input','img','hr','br'];
	$noblock = ['a', 'span', 'img', 'input'];
	$add = '';
	if (is_string($attr) && json_decode($attr, true)) {$attr = json_decode($attr, true);}
	if (is_array($attr)) {foreach ($attr as $k=>$v) {$add .= " $k=\"$v\"";}}
	elseif ($attr) {$add = " $attr";}

	if (in_array($tag, $self_closing)) {$string = "<$tag$add />";}
	else {$string = "<$tag$add>$string</$tag>";}
	if (!in_array($tag, $noblock)) {$string .= "\n";}
	return $string;
	}

function strong($string) {return htmltag('strong', '', $string);}

/*
 * returns info of a file
 * @param string file (path and name)
 * @return array name, mime, size, readable size (psize), icon, perms, owner, group, mtime
 */
function file_info($file) {
	global $html;
	$finfo = finfo_open(FILEINFO_MIME_TYPE);
	$data['mime'] = finfo_file($finfo, $file);
	// more mime than possible
	$info = pathinfo($file);
	$data['name'] = $info['basename'];
	$data['ext'] = isset($info['extension']) ? $info['extension'] : '';
	if (preg_match('/json/', $data['mime'])) {$data['mime'] = "text/json";}
	if ($data['mime'] == "text/plain") {
		if (array_key_exists('extension', $info)) {
			if ($info['extension'] == "css") {$data['mime'] = "text/css";}
			if ($info['extension'] == "js") {$data['mime'] = "text/x-javascript";}
			if ($info['extension'] == "sql") {$data['mime'] = "text/sql";}
			if ($info['extension'] == "pm") {$data['mime'] = "text/perl";}
			}
		}
	$data['size'] = filesize($file);
	$data['psize'] = sizeme(filesize($file));
	//$data['icon'] = get_icon("png-mime/".mime_icon($data['mime']), $data['mime']);
	$data['icon'] = get_icon(mime_icon($data['mime']), strtoupper($data['ext']));
	$data['perms'] = fileperms($file);
	$data['owner'] = fileowner($file);
	$data['group'] = filegroup($file);
	$data['mtime'] = filemtime($file);
	return $data;
	}

/*
 * make readable file-size
 * @param string number of bytes
 * @return string number of KB, MB etc. with largest range
 */
function sizeme($bytes) {
	$sz = ['B', 'KB', 'MB', 'GB', 'TB', 'PB'];
	$factor = floor((strlen($bytes) - 1) / 3);
	return sprintf("%.2f", $bytes / pow(1024, $factor))." ".$sz[$factor];
	}

/*
 * file-size in Bytes
 * @param string 20MB
 * @return string number in Bytes
 */
function size_bytes($val) {
    $val = trim($val);
	$last = strtoupper($val[strlen($val)-1]);
	$val = intval($val);
	switch($last) {
        case 'G': $val *= (1024 * 1024 * 1024); break;
        case 'M': $val *= (1024 * 1024); break;
        case 'K': $val *= 1024; break;
		}
    return $val;
	}

/*
 * converts seconds to hours, second, minutes
 * @param string seconds
 * @return string H:i:s
 */
function calc_seconds($sec) {
	$his = explode(':', gmdate('H:i:s', $sec));
	$out = '';
	if ($his[0] != '00') {$out = $his[0].'h '.$his[1].'min '.$his[2].'sec';}
	elseif ($his[1] != '00') {$out = $his[1].'min '.$his[2].'sec';}
	else {$out = $his[2].'sec';}
	return $out;
	}

/**
 * reads value pairs from a settings table
 * @param1 array $def keys: group (mandantory), table (opt)
 * @return array assoc array with value pairs
 */
function read_settings($def, $ent = true) {
	global $db, $con;
	$settings = array();
	if (!array_key_exists('table', $def)) {$def['table'] = 'cms_settings';}
	$query = "SELECT `key`, `value`, `lang` FROM `{$def['table']}` WHERE `rule` = ? AND `group` = ?;";
	$arg = $db->prepare($query);
	$arg->execute(array($con['rule'], $def['group']));
	while ($result = $arg->fetch(PDO::FETCH_ASSOC)) {
		if ($result['value']) {$val = $ent ? htmlentities($result['value']) : $result['value'];}
		else {$val = '';}
		if ($result['lang'] == "*") {$settings[$result['key']] = $val;}
		else {$settings[$result['key'].'_'.$result['lang']] = $val;}
		}
	return ($settings);
	}

/**
 * returns an icon name for given type
 * @param string the mimetype
 * @return string the icon name
 */

// 'application/pdf' => 'fa-file-pdf-o',
//     'application/msword' => 'fa-file-word-o',
//     'application/vnd.ms-word' => 'fa-file-word-o',
//     'application/vnd.oasis.opendocument.text' => 'fa-file-word-o',
//     'application/vnd.openxmlformats-officedocument.wordprocessingml' => 'fa-file-word-o',
//     'application/vnd.ms-powerpoint' => 'fa-file-powerpoint-o',
//     'application/vnd.openxmlformats-officedocument.presentationml' => 'fa-file-powerpoint-o',
//     'application/vnd.oasis.opendocument.presentation' => 'fa-file-powerpoint-o',





function mime_icon($type) {
	switch ($type) {
		case "directory": $icon = "folder"; break;
		case 'application/vnd.oasis.opendocument.text':
		case 'application/vnd.ms-word':
	    case 'application/vnd.oasis.opendocument.text':
	    case 'application/vnd.openxmlformats-officedocument.wordprocessingml':
		case 'application/vnd.openxmlformats-officedocument.wordprocessingml.document':
		case "application/msword": $icon = "far-file-word"; break;
		case "application/octet-stream": $icon = "cog"; break;
		case "application/pdf": $icon = "far-file-pdf"; break;
		case 'application/vnd.ms-excel':
		case 'application/vnd.openxmlformats-officedocument.spreadsheetml':
	    case 'application/vnd.oasis.opendocument.spreadsheet':
	    case "application/vnd.ms-excel": $icon = "far-file-excel"; break;
		case 'application/vnd.openxmlformats-officedocument.presentationml':
		case 'application/vnd.oasis.opendocument.presentation':
		case "application/vnd.ms-powerpoint": $icon = "far-file-powerpoint"; break;
		case "application/x-shockwave-flash": $icon = "flash"; break;
		case "application/x-tar":
		case "application/zip":
		case "application/gzip": $icon = "far-file-archive"; break;
		case "audio/basic":
		case "audio/midi":
		case "audio/mpeg":
		case "audio/x-wav":
		case "video/mpeg": $icon = "far-file-sound"; break;
		case "video/quicktime":
		case "video/x-msvideo":
		case "video/x-sgi-movie": $icon = "far-file-video"; break;
		case "image/bmp":
		case "image/x-icon":
		case "image/gif":
		case "image/jpeg":
		case "image/svg":
		case "image/svg+xml":
		case "image/png": $icon = "far-file-image"; break;
		case "text/css": $icon = "css3"; break;
		case "text/html": $icon = "html5"; break;
		case "text/php":
		case "text/x-php": $icon = "php"; break;
		case "text/xml":
		case "text/x-javascript":
		case "application/x-javascript":
		case "text/perl":
		case "text/x-perl": $icon = "far-file-code"; break;
		case "text/plain":
		case "text/rtf": $icon = "far-file-text"; break;
		case "text/sql": $icon = "sql"; break;
		case "text/xml": $icon = "xml"; break;
		default: $icon = "far-file";
		}
	return $icon;
	}

function get_syntax($type) {
	switch ($type) {
		case "text/css": $mode = "css"; break;
		case "application/x-javascript":
		case "text/x-javascript": $mode = "javascript"; break;
		case "text/php":
		case "text/x-php": $mode = "php"; break;
		case "text/perl":
		case "text/x-perl": $mode = "perl"; break;
		case "text/sql": $mode = "sql"; break;
		default: $mode = "text/html";
		}
	return $mode;
	}

// a public form should always have a get
// before post request. a get delivers a tan. a post requires
// a valueable tan. beetween get and post should be
// a few seconds delay. this prevents silly bots doing
// silly things.
function get_before_post($delay) {
	global $db;
	$err = 0;
	$tan = param('tan') ?  clean_var(param('tan'), 'wr') : gen_tan(12);
	if ($_SERVER['REQUEST_METHOD'] == "GET") {
		$arg = $db->prepare("INSERT INTO `get_before_post` SET `tan` = ?, `ip` = ?;");
		$arg->execute([$tan, $_SERVER['REMOTE_ADDR']]);
		}
	elseif ($_SERVER['REQUEST_METHOD'] == "POST") {
		$arg = $db->prepare("SELECT `ip`, `get_stamp` FROM `get_before_post` WHERE `tan` = '$tan';");
		$arg->execute();
		$data = $arg->fetch(PDO::FETCH_ASSOC);
		if (isset($data['ip'])) {
			$post_stamp = date('Y-m-d H:i:s');
			$arg = $db->exec("UPDATE `get_before_post` SET `post_stamp` = '$post_stamp' WHERE `tan` = '$tan'");
			if ($data['ip'] != $_SERVER['REMOTE_ADDR']) {$err = 1;}
			$seconds = strtotime($post_stamp) - strtotime($data['get_stamp']);
			if ($seconds < $delay) {$err = 2;}
			}
		else {$err = 1;}
		}
	else {$err = 1;}
	// clean house
	$del = date("Y-m-d H:i:s", time() - (3600*48));
	$arg = $db->exec("DELETE FROM `get_before_post` WHERE `get_stamp` < '$del';");
	if ($err == 1) {error_log("Consentio Notice: POST without GET", 0);}
	return array($tan, $err);
	}

# remove tan after sent data **DEPRECATED**
function  remove_tan() {
	// global $db;
	// $tan = clean_var(param('tan'), 'wr');
	// $arg = $db->exec("UPDATE `get_before_post` SET `tan` = '' WHERE `tan` = '$tan'");
	}

function check_extension($filename, $ext) {
	$info = pathinfo($filename);
	if (!array_key_exists('extension', $info)) {$filename .= ".$ext";}
	return $filename;
	}

/**
 * just create a funny tan
 * @param integer number of digits
 * @return string a funny tan
 */
function gen_tan($i) {
	$tan = '';
	for ($j=0; $j<$i; $j++) {
		$rand = intval(rand(0,35));
		if ($rand > 9) {$tan .= chr(87+$rand);}
		else {$tan .= $rand;}
		}
	return ($tan);
	}

/**
 * get a record of a table
 * @param integer id of record
 * @param string name of table
 * @return array assoc array of record
 */
function row_data($id, $table) {
	global $db;
	$arg = $db->prepare("SELECT * FROM `$table` WHERE `id` = ?;");
	$arg->execute(array($id));
	$data = $arg->fetch(PDO::FETCH_ASSOC);
	return $data;
	}

/**
 * update a single value
 * @param integer id of record
 * @param string name of table
 * @param string name of field
 * @param string value
 * @return integer affected row
 */
function update_row($id, $table, $field, $val) {
	global $db;
	$arg = $db->prepare("UPDATE `$table` SET `$field` = ? WHERE `id` = ?;");
	$arg->execute(array($val, $id));
	return $arg->rowCount();
	}

/**
 * export query to xml with ExcelWriterXML
 * @param1 string sql select-statement
 * @param2 array execute parameters (tainted query)
 * @param3 string name of sheet
 * @returns xml file as output
 */
function export_xml($q, $t, $name) {
	require_once ('exporter_lib.php');
	$export = new exporter_lib(['name' => $name, 'query' => $q, 'taint' => $t]);
	$export->write();
	}

/**
 * export table xml with ExcelWriterXML
 * @param1 array row data from $access->table_rows
 * @param2 string name of sheet
 * @returns xml file as output
 */
function export_table_xml($rows, $name) {
	require_once ('exporter_lib.php');
	$export = new exporter_lib(['name' => $name, 'rows' => $rows]);
	$export->write();
	}

function export_xls($q, $t, $name) {
	require_once ('exporter_lib.php');
	$export = new exporter_lib(['format' => 'xls', 'name' => $name, 'query' => $q, 'taint' => $t]);
	$export->write();
	}

function export_table_xls($rows, $name, $capt = false) {
	require_once ('exporter_lib.php');
	$export = new exporter_lib(['format' => 'xls', 'name' => $name, 'rows' => $rows, 'captions' => $capt]);
	$export->write();
	}

function export_xlsx($q, $t, $name) {
	require_once ('exporter_lib.php');
	$export = new exporter_lib(['name' => $name, 'query' => $q, 'taint' => $t]);
	$export->write();
	}

	function export_table_xlsx($rows, $name, $capt = false) {
		require_once ('exporter_lib.php');
		$export = new exporter_lib(['name' => $name, 'rows' => $rows, 'captions' => $capt]);
		$export->write();
		}

function export_csv($q, $t, $name) {
	require_once ('exporter_lib.php');
	$export = new exporter_lib(['format' => 'csv', 'name' => $name, 'query' => $q, 'taint' => $t]);
	$export->write();
	}

function stan($a) {
	$c = implode(';', $a);
	$c = base64_encode($c);
	$c = str_rot13($c);
	return $c;
	}

function ollie($s) {
	$a = [];
	if (substr($s, -1) == '=') {
		$s = str_rot13($s);
		$s = base64_decode($s);
		$a = explode(';', $s);
		}
	return $a;
	}

function yololo ($p) {
	global $con;
	return password_hash($p, PASSWORD_BCRYPT, ['cost' => 12]);
	}

?>
